#V10: This version cleans up the code for liberation. Its based on V08.

############################################################
# Code for cages. V10
#
# GenOne(d,g,n) 
#     Tries to generate (d,g)-graph of order n.
#
# GenAll(d,g,n) 
#     Generates all (d,g)-graphs of order n.
#
# Cage(d,g) 
#     Generates a (d,g)-cage trying to generate 
#     (d,g)-graphs of order n, starting with n=MooreBound(d,g) 
#     and increasing n whenever that fails. 
#
# Cages(d,g) 
#     Generates all (d,g)-cages trying to generate 
#     (d,g)-graphs of order n, starting with n=MooreBound(d,g) 
#     and increasing n whenever that fails. 
#############################################################

#State record.
if not IsBound(state) then
  state:=rec();
fi;

LoadPackage("yags");
Read("isonauty.g");
Read("progress.g");
Read("stab.g");
Read("choose.g");
Read("cases.g");
Read("bounds.g");
########################### timeout #################################
if not IsBound(timeout) then
  DeclareCategory("timeoutcat",IsObject);
  InstallMethod(PrintObj,"for timeout",true,[timeoutcat],0,
    function(timeout)Print("timeout");end);
  DeclareGlobalVariable("timeout");
  InstallValue(timeout,Objectify(NewType(NewFamily("timeoutfamily"),timeoutcat),rec()));
fi;
######################################################################

#State record, defaults.
state0:=rec(dgn:=[],NumSols:=0,Cases:=0,Sizes:=0,Progress:=0);
state0.format := [ "V10, ","dgn", "Suffix", "Ex", "ExG", "Datadir","\n",
                   "Cases", "NodeCount","NPS","NumSols", "\n",
                   "ET", "ETC", "ETCR","Progress" ];
state0.mb:=0; #MooreBound(n,g);
state0.Ex:=0; #Number of ExcessVertices
state0.EX:=[]; #ExcessVertices: [r.n-r.Ex+1..r.n] 
state0.ExG:=0; #ExcessGraphs 
state0.Part:=[]; #Excess vertices partition: [state0.EX];
state0.d1:=0;state0.d2:=0;state0.d3:=0; #For girth testing.
state0.Datadir:="Data";
###
state0.LG:=[]; #List of feasible solutions.
state0.LSol:=[]; #List of solutions.
###
state0.GlobalTimers:=[]; #GlobalTimers;
state0.Remaining:=[1]; #Remaining;
state0.ET:=0;
state0.ETC:="--";
state0.ETCR:="--";
state0.History:=[[0,0,0]];#History. [et,prog,solved nodes]
state0.HistLen:=20;
state0.FProgress:=Progress; #Progress as a fraction.
state0.Progress:=0.; #Progress as a Float. 
state0.times:=[]; #times for solutions and total time.
state0.NodeCount:=0;
state0.NPS:=0; #solved nodes per second.
###############################################################
### Efficiency parameters:

# Suffix added to filename to distinguish experiments 
# with different efficiency parameters. Computed in Initialize().
state0.Suffix:="";

# Use Lnew reduction in ProcessGraphs for reducing cases 
# when two new graphs are isomorphic. This happens occasionally 
# but its costly to check apparently.
# default: false, V05 =true, V06=false.
state0.useLnewreduce:=false;
#state0.useLnewreduce:=true; #Bad one, do not use.

# Use PreviouslyConsidered1 instead of PreviouslyConsidered 
# PrevCons1 is the original version too slow for many large cases. 
#           May produce OutOfMemory conditions.
# PrevCons is the better "tapahoyos" algorithm.
# when (S choose k) < 10^state.usePrevCons1.
# default: 3 (10^3), V05 = 0, V06=3. 
#   -- we also tried 2^r for r=9,10,11 and 10^r for r=0,1,2,4, and others,
#   -- all of them slower or much slower.
#state0.usePrevCons1:=0; #never use PrevCons1. Slow for [7,6,88] and others.
state0.usePrevCons1:=16;  #(=r) use PrevCons1 when (S choose E) < 2^r.
#state0.usePrevCons1:=11;  #Only 12% slower that 16. Perhaps avoids OutOfMemory Conditions.
#state0.usePrevCons1:=infinity; #always use PrevCons1. Too slow for [8,5,66+] and many others.

# Use nauty to compute all possible excess graphs 
# upto isomorphism for vertices beyond the Moore Bound.
# 'false' is too slow for some [3,10,68+] (18.3s vs 10min) and many others.
# specially combined with state.useparenttrivial = 1. (18.3s vs 1.13h)
#default: true, V05 = true, V06=true.
state0.useexcess:=true;
#state0.useexcess:=false; #Bad one, do not use.

# Do not use or compute the automorphism group when 
# many (>=state.useparenttrivial) ancestor graphs had trivial group.
# The AutGroup is computed anyway if state.useLnewreduce=true.
# default: infinity, V05 = infinity V06 = 1.
#Different values are much slower than this two values:
#state0.useparenttrivial:=1; #Second best.
state0.useparenttrivial:=infinity; #Best.
###

Timeout:=infinity; #seconds.
  

######################################################
##
## Sustitute for girth.
##
######################################################

AdjD1:=function(G,x,r)
  local M,h,lis,blis,uni,z;
  M:=AdjMatrix(G);
  uni:=[1..Order(G)];
  if not IsBound(G!.adjd1) then
     G!.adjd1:=[];
  fi;
  if IsBound(G!.adjd1[x]) then
    return G!.adjd1[x];
  fi;
  G!.adjd1[x]:=BlistList(uni,[x]);
  for h in [1..r.d1] do
    lis:=ListBlist(uni,G!.adjd1[x]);
    for z in lis do
       UniteBlist(G!.adjd1[x],M[z]);
    od;
  od;
  return G!.adjd1[x];
end;

AdjD3:=function(G,x,r)
  local M,h,lis,blis,uni,z;
  M:=AdjMatrix(G);
  uni:=[1..Order(G)];
  if not IsBound(G!.adjd3) then
     G!.adjd3:=[];
  fi;
  if IsBound(G!.adjd3[x]) then
    return G!.adjd3[x];
  fi;
  G!.adjd3[x]:=ShallowCopy(AdjD1(G,x,r));
  lis:=ListBlist(uni,G!.adjd3[x]);
  for z in lis do
    UniteBlist(G!.adjd3[x],M[z]);
  od;
  return G!.adjd3[x];
end;

AdjD2:=function(G,x,r)
   if r.d2=r.d3 then
     return AdjD3(G,x,r);
   else;
     return AdjD1(G,x,r);
   fi;
end;

# Adds elements from  LG to LSol 
# upto equivalence (isomorphism).
# It is like UniteSet but checking 
# equivalence instead of equality.
AddEquiv:=function(LSol,LG,Equiv) 
  local G;
  for G in LG do
    if ForAll(LSol,z->not Equiv(z,G)) then 
      Add(LSol,G);
    fi;
  od;
end;


ExcessGraphs:=function(r) 
  local str,out,d,g,m,G,tempfile;
  d:=r.d;g:=r.g;m:=r.n-r.mb;
  if not r.useexcess then 
    r.Ex:=0; 
    G:=DiscreteGraph(m); 
    return [G]; 
  fi;
  r.Ex:=m;
  if m=1 then return [DiscreteGraph(1)]; fi; #Graph6ToGraph bugfix.
  if m>32 then
    r.Ex:=32;
  fi; 
  r.EX:=[r.n-r.Ex+1..r.n];
  r.Part:=[r.EX];
  #Exec("geng -D5tf 4 | pickg -~g1:4 > tempd.g.n.g6");
  tempfile:=StringFormatted("{}/temp{}.{}.{}.{}.g6",r.Datadir,r.d,r.g,r.n,r.Suffix);
  str:="geng -D";Append(str,String(d)); #MaxDegree = d
  if g>=4 then Append(str,"t");fi; #No triangles.
  if g>=5 then Append(str,"f");fi; #No squares.
  Append(str," ");
  Append(str,String(r.Ex)); #Order
  Append(str," | pickg -~g1:");
  Append(str,String(g-1));
  Append(str," > ");
  Append(str,tempfile);
  Print(">A ",str,"\n");
  Exec(str);
  out:=ImportGraph6(tempfile);
  str:=StringFormatted("rm {}",tempfile);
  Exec(str);
  r.ExG:=Length(out);
  if m>r.Ex then 
    List(out,g->DisjointUnion(DiscreteGraph(m-r.Ex),g));
  fi;
  Print(">Z ",Length(out)," excess graphs.\n");
  return Reversed(out);  
  #return out;
end;

MooreTree:=function(d,g)
   local L,G,G1,G2;
   if IsOddInt(g) then
      L:=List([1..(g-1)/2],z->d-1);
      L[1]:=d;
      G:=TreeGraph(L);
   else # Even g
      L:=List([1..g/2-1],z->d-1);
      G1:=TreeGraph(L);
      G2:=TreeGraph(L);
      G:=DisjointUnion(G1,G2);
      G:=AddEdges(G,[[1,Order(G1)+1]]);
   fi;
   return G; 
end;


#####################################
#  MinOrder(d,g)
#  MinOrder(d,g,n0)
#
# Returns n where n is the minimum order 
# for a possible (d,g)-graph which is at least n0. 
#
MinOrder:=function(arg)
  local d,g,n0,n,step;
  d:=arg[1];g:=arg[2];
  n0:=0;
  if IsBound(arg[3]) then 
    n0:=arg[3];
  fi;
  n:=Maximum(n0,MooreBound(d,g));
  if IsOddInt(d) and IsOddInt(n) then n:=n+1; fi;
  return n;
end;


##############################################
Candidates:=function(G,r) 
  return Filtered(Vertices(G),x->VertexDegree(G,x)<r.d);
end; 

Conflicts:=function(G,cands,r)
  local grp,confs,equiv;
  grp:=auto2(G,r.Part);
  equiv:=function(x,y) return Transporter(grp,x,y)<>fail; end;
  confs:=EquivalenceRepresentatives(cands,equiv);
  return confs;
end;

##########################################################
## Returns the graph G with all edges from x to sol added. 
##########################################################

ApplySol:=function(G,x,sol) 
   return AddEdges(G,Cartesian([x],sol));
end;

##################################################################################

CandidatesForC:=function(G,cands,c,r)
   local candsc;
   if c in r.EX then 
     candsc:=Filtered(cands,z->not z in r.EX);
   else
     candsc:=cands;
   fi;
   candsc:=Filtered(candsc,z->SizeBlist(IntersectionBlist(AdjD3(G,c,r),AdjD2(G,z,r)))=0);
   return candsc;
end;

ChooseConflict:=function(G,grp,confs,cands,r)
  local check,sol,grp1,sols,Lindices,i,j,c,chosen,candsc,k,L,d; 
  
  d:=r.d;
  
  check:=function(L)
    local y,z,G1,blist; 
    if L=[] then return true; fi;
    y:=L[Length(L)];
    if c=y then return false; fi;
    for z in L{[1..Length(L)-1]} do
      blist:=IntersectionBlist(AdjD2(G,y,r),AdjD1(G,z,r));
      if SizeBlist(blist)>0 then return false; fi;       
    od;
    return true;   
  end;
  
  Lindices:=List(confs,z->[]);
  sols:=List(confs,z->[]);
  i:=0; #i= number of solutions for special tratment when i=1.
  while(true) do
      i:=i+1; 
      for j in [1..Length(confs)] do
         c:=confs[j];
         grp1:=yStab(grp,c);
         candsc:=CandidatesForC(G,cands,c,r);
         k:=d-VertexDegree(G,c);
         L:=Lindices[j];
         sol:=ChooseNext(candsc,k,grp1,check,L);
         if sol = fail and i=1 then                
             return fail;
         elif sol=fail then #(i>1). select this conflic to solve.
            return [c,sols[j]];
         fi;
         Add(sols[j],sol);
         if i=1 then #compute a second solution.
           sol:=ChooseNext(candsc,k,grp1,check,L);
           if sol=fail then #Only one solution, use this one. 
             return [c,sols[j]]; 
           fi;
           Add(sols[j],sol);
         fi;
      od;
  od;
end;

#NodesPerSecond
NPS:=function(r) 
  local len,DeltaT,DeltaCount;
  if Length(r.History)<2 then return "--"; fi;
  len:=Length(r.History);
  DeltaT:=r.History[len][1]-r.History[1][1];
  if DeltaT<=0 then return "--"; fi;
  DeltaCount:=r.History[len][3]-r.History[1][3];
  return Int(DeltaCount*10^9/DeltaT);
end;

#Complete all entries of record r and report.
#called only when it is actually going to be reported.
ReportNow:=function(r) 
  r.nET:=NanoElapsedTime();#in nanoseconds.
  r.ET:=THumanTime(r.nET);#in human-readable form.
  r.Cases:=Length(r.LG);
  r.FProgress:=Progress;
  r.Progress:=Float(r.FProgress);
  r.Remaining:=Remaining;
  if Length(r.History)>= r.HistLen then
    Remove(r.History,1);
  fi;
  Add(r.History,[r.nET,Progress,r.NodeCount]);
  r.ETC:=ETC(1,r.nET);
  r.ETCR:=ETCR(r.History);
  r.NPS:=NPS(r);
  r.GlobalTimers:=GlobalTimers;
  if r.LG<>[] then
   r.Sizes:=Size(r.LG[Length(r.LG)]);
  fi;
  ReportRec(r);
end;

#Stores record on disk.
StoreState:=function(r) 
  local file1,file2,cmd;
  ReportNow(r); #Update all fields in record before storing.
  file1:=StringFormatted("{}/state{}.{}.{}.{}.g",r.Datadir,r.d,r.g,r.n,r.Suffix);
  file2:=Concatenation(file1,".bak");
  cmd:=StringFormatted("test -f {} && cp {} {}",file1,file1,file2);
  Exec(cmd);
  PrintTo(file1,"state:=",r,";\n",
     "Remaining:=state.Remaining;\n",
     "Progress:=state.FProgress;\n",
     "GlobalTimers:=state.GlobalTimers;\n",
     "usePrevCons1:=state.usePrevCons1;\n");
  
  cmd:=StringFormatted("test -f {} && rm {}",file2,file2);
  Exec(cmd);
end;

#Reports and other things
#Once Per Second.
OPS:=function(r) 
  local et;
  et:=NanoElapsedTime(3); #resets every second.
  if et < 10^9 then 
    return;
  fi;
  ReportNow(r); 
  ResetTimer(3);
  et:=NanoElapsedTime(4); #resets every minute.
  if et< 60*10^9 then 
    return;
  fi;
  StoreState(r); #this takes about 12 ms.
  ResetTimer(4);
end;

#########################################################################
# Removes one graph from LG and generate its subcases and append them to LG.
# Updates progress and state, and reports.
#
# Returns G1 if a new solution found.
# Returns true if there are more cases to analyze.
# Returns fail when LG=[].
ProcessGraphs:=function(r) 
  local G1,confs,sols,Lnew,Lnew2,c,cands,grp,
        sol,chosen,LG,d,g,n,newsol,et,gg; 
  d:=r.d;g:=r.g;n:=r.n;
  newsol:=false;
  LG:=r.LG;
  if LG=[] then #No more solutions. 
    Add(r.times,ElapsedTime());
    StoreState(r);
    return fail; 
  fi; 
  G1:=LG[Length(LG)];#Do not remove yet.
  if not IsBound(G1!.parenttrivialgroup) then 
    G1!.parenttrivialgroup:=0; 
  fi; 
  cands:=Candidates(G1,r); #vertices of degree < d
  if  G1!.parenttrivialgroup >= r.useparenttrivial then
    grp:=Group([()]);
    confs:=cands;
  else 
    #grp:=AutomorphismGroup(G1); 
    grp:=auto2(G1,r.Part);
    confs:=Conflicts(G1,cands,r);#conflict vertices mod Aut(G).
  fi;
  Lnew:=[];Lnew2:=[];
  sol:=true;#default: keep trying.
  if confs=[] then #### Solution. 
    AddEquiv(r.LSol,[G1],isiso);
    if r.NumSols<>Length(r.LSol) then 
       r.NumSols:=Length(r.LSol);
       sol:=G1; 
       Add(r.times,ElapsedTime());
       newsol:=true;
    fi;
  else  
    chosen:=ChooseConflict(G1,grp,confs,cands,r);
    if chosen<>fail then #Else is unsolvable case.
      c:=chosen[1];
      sols:=chosen[2];  
      Lnew:=List(sols,sol->ApplySol(G1,c,sol));
      #Perform(Lnew,function(z) z!.DreadPartition:=G1!.DreadPartition; end);
      if r.useLnewreduce then
        AddEquiv(Lnew2,Lnew,{g,h}->isiso2(g,h,r.Part));#occasionally two new cases are isomorphic. 
      else 
        Lnew2:=Lnew;
      fi;
    fi;
  fi;
  for gg in Lnew2 do 
    if IsTrivial(grp) then
      gg!.parenttrivialgroup:=G1!.parenttrivialgroup+1;
    else
      gg!.parenttrivialgroup:=0;
    fi;
  od;
  ####### Reporting
  Remove(LG);
  Append(LG,Lnew2); 
  UpdateProgress(Length(Lnew2));
  r.NodeCount:=r.NodeCount+1;
  OPS(r);
  if newsol then StoreState(r); fi; #new solution.
  ##########
  return sol; 
end; #end ProcessGraphs().

#####################################
#Initialize(rec)
#Initialize(d,g,n)
#Initialize(d,g,n,LG) #FIXME: to be programmed.
#
#Initializes and returns r record.
#
Initialize:=function(arg)
  local G,r,sfx,gg;
  ResetTimer(3);
  ResetTimer(4);
  if Length(arg)= 1  then
    # arg  is already a record. Adjust timer
    # for possible interruptions.
    r:=arg[1];
    AdjustTimer();
    return r;
  fi;
  ########## d,g,n,step,mb,Timer,Suffix.
  ResetTimer();
  r:=StructuralCopy(state0); #defaults
  r.GlobalTimers:=GlobalTimers;#defined in progress.g.
  r.Remaining:=Remaining;#defined in progress.g.
  r.d:=arg[1];r.g:=arg[2];r.n:=arg[3];
  r.mb:=MooreBound(r.d,r.g);
  if IsOddInt(r.d) then r.step:=2; else r.step:=1; fi;
  r.dgn:=[r.d,r.g,r.n];
  r.d3:=Int((r.g+1)/2)-1; #  Ceil((g-2)/2)
  r.d2:=Int(r.g/2)-1;     # Floor((g-2)/2) =  Ceil((g-3)/2)
  r.d1:=r.d3-1;           #                = Floor((g-3)/2)
  sfx:=r.Suffix;
  if r.useLnewreduce then 
    Append(sfx,"T"); 
  else 
    Append(sfx,"F"); 
  fi;
  if r.usePrevCons1=infinity then 
    Append(sfx,"I"); 
  else 
    Append(sfx,String(r.usePrevCons1));
  fi;
  if r.useexcess then Append(sfx,"T"); else Append(sfx,"F"); fi;
  if r.useparenttrivial=infinity then 
    Append(sfx,"I"); 
  else 
    Append(sfx,String(r.useparenttrivial));
  fi;
  r.Suffix:=sfx;
  ########## Defined in sgp2.g:
  usePrevCons1:=r.usePrevCons1;
  ########## LG: initial cases.
  G:=MooreTree(r.d,r.g);
  if Order(G)<> r.mb then Error("MooreBound<>Order(MooreTree)\n"); fi;
  if Order(G)=r.n then
    r.LG:=[G];  
  elif Order(G)>r.n then 
    r.LG:=[];
  else #Order(G)<r.n
    r.LG:=ExcessGraphs(r);
    r.LG:=List(r.LG,z->DisjointUnion(G,z));
  fi;
  r.LSol:=[];
  ########## Progress
  ResetProgress();
  UpdateProgress(Length(r.LG));
  return r;
end; #end Initialize()

############################################
# Usage:
# GenOne(d,g,n)
# GenOne(state)
# 
# Generates one (d,g)-graph on n vertices.
#
# Returns (d,g)-graph if found.
# Returns fail if there is no such (d,g)-graph.
# Returns timeout if NanoElapsedTime()> Timeout*10^9.
GenOne:=function(arg)
  local sol;
  state:=CallFuncList(Initialize,arg);
  if state=fail then return fail; fi;
  StoreState(state);
  sol:=true;
  while(sol=true) do 
    sol:=ProcessGraphs(state);
    if Graphs(sol) then 
      StoreState(state);
      Report("*** SOLUTION ***\n");
      return sol;
    elif sol=fail then
      StoreState(state);
      Report("*** No More Cases ***\n");
      return fail;
    elif NanoElapsedTime() > Timeout*10^9 then 
      StoreState(state);
      Report("*** TIMEOUT ***\n");      
      return timeout;
    fi;  
  od;
end;

##############################################
# Usage:
# GenAll(d,g,n)
# GenAll(state)
# 
# Generates all (d,g)-graph on n vertices.
#
# Returns list of (d,g)-graph if found.
# Returns timeout if NanoElapsedTime()> Timeout*10^9.
GenAll:=function(arg)
  local sol;
  state:=CallFuncList(Initialize,arg);
  if state=fail then return []; fi;
  sol:=true;
  while(not sol in [fail,timeout]) do 
    sol:=GenOne(state);
  od;
  if sol=timeout then 
    return timeout; 
  else
    Report("*** No More Solutions ***\n");
    return state.LSol;  
  fi;
end;


############################################
# usage:
# Cage(d,g)
# Cage(d,g,n)
# Cage(state)
# 
Cage:=function(arg)  
  local sol,d,g,n,step,nstep;
  if Length(arg)>1 then
    arg[3]:=CallFuncList(MinOrder,arg);
  fi;  
  state:=CallFuncList(Initialize,arg);  
  ResetTimer(2);
  sol:=fail;
  while sol=fail do
    sol:=GenOne(state);
    if sol=timeout or NanoElapsedTime(2)>Timeout*10^9 then
      StoreState(state);
      return timeout; 
    fi;
    if sol<>fail then break; fi;
    arg:=[state.d,state.g,state.n+state.step];
    state:=CallFuncList(Initialize,arg);  
  od; 
  StoreState(state);
  Report("**** CAGE ****\n");      
  return sol;
end;

############################################
# usage:
# Cages(d,g)
# Cages(d,g,n)
# Cages(state)
# 
Cages:=function(arg) 
  CallFuncList(Cage,arg);
  GenAll(state);
  #StoreState(state);
  return state.LSol;
end;


######################################################
#Prints Degrees, girth and order of G, when G is a Graph.
#For sanity checks.
CheckSol:=function(G)
  if Graphs(G) then 
    Print(Set(VertexDegrees(G)),"\n",Girth(G),"\n",Order(G),"\n");
  fi;  
end;

GenAllCases:=function(Lcases) 
  local case,a,b,c,d; #F3TI is the reference. 
  ResetTimer(5);
  for case in Lcases do 
     CallFuncList(GenAll,case);
     if NanoElapsedTime(5)> Timeout*10^9 then
       Print("Timeout!\n");
       break;
     fi;   
  od;
end;


Lf:=[GenOne,ProcessGraphs,Candidates,Conflicts,CandidatesForC,ChooseConflict,ChooseNext,AdjD1,AdjD2,AdjD3,ApplySol,auto,yStab];
