#Interface to drednaut. 
#Used for AutomorphismGroup, IsIsomorphicGraph,
# CanonicalLabeling and Hash.
DeclareAttribute("CanonizedGraph6",Graphs);
DeclareAttribute("CanonicalLabeling",Graphs);
DeclareAttribute("Hash",Graphs);

if not IsBound(dreadstrm) then 
  dreadstrm:=fail;
fi;

#forward declaration
DreadStart:=function() end;

#Write and read from dreadnaut.
DreadAsk:=function(str)
  local out,len,res,strout;
  out:=[];
  res:=WriteLine(dreadstrm,str);
  # if res=fail then #retry once #FIXME: not working. Signals an error anyway.
  #   DreadStart();
  #   res:=WriteLine(dreadstrm,str);
  #   if res=fail then return fail; fi;
  # fi;
  len:=Length(out);
  while(len<2 or out{[len-1,len]}<>"> ") do
    strout:=ReadAll(dreadstrm);
    if strout=fail then break; fi;
    Append(out,strout);
    len:=Length(out);
  od;
  return out;
end;

#Initialize dreadnaut.
DreadStart:=function()
  local wd,sd,dp,str,out,len;
  out:="";
  sd:=DirectoriesSystemPrograms();
  wd:=DirectoryCurrent();
  dp:=Filename(sd, "dreadnaut"); 
  dreadstrm:=InputOutputLocalProcess( wd, dp, [] );
  if dreadstrm=fail then 
    Print("#I Unable to start dreadnaut process.");
    return fail;
  fi;
  len:=Length(out);
  while(len<2 or out{[len-1,len]}<>"> ") do
    Append(out,ReadAll(dreadstrm));# discard initial response.
    len:=Length(out);
  od;
  str:="l=0$=1dac";
  Append(out,DreadAsk(str));
  return [str,out];
end;

if dreadstrm=fail then DreadStart(); fi;

#Load graph G to dreadnaut.
DreadLoadG:=function(G) 
  local str,out,x,y;
   str:="";
   Append(str,"n=");
   Append(str,String(Order(G)));
   Append(str," g");
   for x in Vertices(G) do
     Append(str," ");
     Append(str,String(x));
     Append(str,":"); 
     for y in Adjacency(G,x) do
       Append(str,String(y));
       Append(str,","); 
     od;
   od;
   Append(str,".");
   out:=DreadAsk(str);
   return [str,out];
end;

#S:=Set of additional vertices to be fixed.
DreadFixSetAdd:=function(S)
  local str,out,s;
  str:="";
  for s in S do
    Append(str," -F=");
    Append(str,String(s));
  od;
  out:=DreadAsk(str);
  return [str,out];
end;

#S:=Set of vertices to be fixed.
DreadFixSet:=function(S) 
  DreadAsk(" -f ");
  return DreadFixSetAdd(S);
end;

# Selects a partition in dreadnaut.
# Example:
# P= [[1,4,3],[2,5,7],[6,8]]
# strp= "f=[1,4,3|2,5,7|6,8]";
DreadPartition:=function(P) 
  local strp,out,i,j;
  strp:="f=[";
  for i in [1..Length(P)] do 
    for j in [1..Length(P[i])] do 
      Append(strp,String(P[i][j]));
      if j<Length(P[i]) then 
        Add(strp,',');
      elif i<Length(P) then
        Add(strp,'|');
      fi;
    od;
  od;
  Add(strp,']');
  out:=DreadAsk(strp);
  return out;
end;

#Uses a DFA to transform dreadnaut output 
#to a list of permutations (generators).
DreadPermDFA:=function(str) 
  local out,i,c,state;
  out:="[";i:=1;state:=0;
  c:=str[i];
  while true do
    if state=0 then #beginning
      if c='(' then
        Add(out,c);
        state:=1;
      elif c='>' then
        Add(out,']');
        break; #done parsing
      else
        state:=2;
      fi;
    elif state=1 then #reading perm
      if c=' ' then
        Add(out,',');
      elif c='\n' then 
        Add(out,',');
        state:=0;
      else 
        Add(out,c);
      fi;
    elif state=2 then #skipping to end of line.
      if c='\n' then
        state:=0;
      #else: do nothing.  
      fi;
    fi;
    i:=i+1;
    c:=str[i];
  od;
  out:=EvalString(out);
  if out=[] then out:=[()]; fi;
  return out;
end;

#Uses a DFA to transform dreadnaut output 
#to a Canonical Labelling (list of vertices).
DreadCanLabDFA:=function(str) 
  local out,i,c;
  out:="[";i:=2;
  c:=str[i];
  while true do #DFA with only one state.
    if c=' ' then
      Add(out,',');
    elif c='\n' then
      Add(out,']');
      break; #done parsing
    else
      Add(out,c);
    fi;
    i:=i+1;
    c:=str[i];
  od;
  out:=EvalString(out);
  return out;
end;

#Perhaps this is faster?
DreadCanLabDFA1:=function(str) 
  local pos,out;
  pos:=Position(str,'\n');
  str:=str{[1..pos]};
  str[1]:='[';
  str[pos]:=']';
  str:=ReplacedString(str," ",",");
  out:=EvalString(str);
  return out;
end;

#Uses a DFA to transform dreadnaut output 
# into a Hash (list of three hexadecimals).
DreadHashDFA:=function(str) #one state DFA.
  local out,i,c;
  out:="[\"";i:=3;
  c:=str[i];
  while true  do
    if c=' ' then
      Append(out,"\",\"");
    elif c=']' then
      Append(out,"\"]");
    elif c='\n' then
      break;
    else 
      Add(out,c);
    fi;
    i:=i+1;
    c:=str[i];
  od;  
  out:=EvalString(out);
  out:=List(out,IntHexString);
  return out;
end;

Dread:=function(G) 
  local str,perms,canlab,hash;
  DreadLoadG(G);
  perms:=DreadPermDFA(DreadAsk("x"));
  canlab:=DreadCanLabDFA(DreadAsk("b"));
  hash:=DreadHashDFA(DreadAsk("z"));
  SetAutomorphismGroup(G,Group(perms)); 
  SetCanonicalLabeling(G,canlab);
  SetHash(G,hash);
  return [perms,canlab,hash];
end;

# DreadWithS:=function(G,S) 
#    DreadLoadG(G);
#    DreadFixSet(S);
#    return Group(DreadPermDFA(DreadAsk("x")));
# end;

isiso:=function(g,h)
    local canlab1,canlab2;
    if Order(g)<> Order(h) then return false; fi;
    if Collected(VertexDegrees(g))<>Collected(VertexDegrees(h)) then 
        return false;
    fi;
    if not HasHash(g) then Dread(g); fi;
    if not HasHash(h) then Dread(h); fi;
    if Hash(g)<>Hash(h) then return false; fi;
    #canlab1:=CanonicalLabeling(g);
    #canlab2:=CanonicalLabeling(h);
    #return InducedSubgraph(g,canlab1)=InducedSubgraph(h,canlab2);    
    return CanonizedGraph6(g)=CanonizedGraph6(h);
end;

auto:=function(G)
   if not HasAutomorphismGroup(G) then 
      Dread(G);
   fi;
   return AutomorphismGroup(G); 
end;                   

cang6:=function(G) 
  local H,canonized;
  if not HasCanonizedGraph6(G) then
     if not HasCanonicalLabeling(G) then 
       Dread(G); 
     fi;
     H:=InducedSubgraph(G,CanonicalLabeling(G));
     canonized:=GraphToGraph6(H);
     SetCanonizedGraph6(G,canonized);
   fi;
   return CanonizedGraph6(G); 
end;

if dreadstrm<>fail then 
  InstallOtherMethod(AutomorphismGroup,"for Graphs",true,[Graphs],10,auto);  
  InstallMethod(IsIsomorphicGraph,"for Graphs",true,[Graphs,Graphs],10,isiso);  
  InstallMethod(CanonizedGraph6,"for Graphs",true,[Graphs],10,cang6);
fi;

####################################
#
# With Partitions
#
###################################

auto2:=function(G,P)
  local perms,canlab;
  if P=[] then return auto(G);fi;
  if IsBound(G!.auto2) then 
    return G!.auto2;
  fi;
  DreadLoadG(G);
  DreadPartition(P);  
  perms:=DreadPermDFA(DreadAsk("x"));
  G!.auto2:=Group(perms);
  G!.canlab:=DreadCanLabDFA(DreadAsk("b"));
  return G!.auto2;
end;

canonize2:=function(G,P) 
  if P=[] then return CanonizedGraph6(G); fi;
  if not IsBound(G!.canonize2) then 
    if not IsBound(G!.canlab) then 
      auto2(G,P);
    fi;
    G!.canonize2:=GraphToGraph6(InducedSubgraph(G,G!.canlab));
  fi;
  return G!.canonize2;
end;

isiso2:=function(G,H,P) 
  return canonize2(G,P)=canonize2(H,P);
end;


#########################################################################
# echo "l=0$=1dacn=5 g1,2,3. xbz" | dreadnaut | grep -v ':' | grep -v ';'
# echo "l=0$=1dacn=5 g 1:2,5 2:1,3 3:2,4 4:3,5 5:1,4. xbz"
#init: "l=0$=1da" ##with Automorphisms.
#init: "l=0$=1dac" ##with Automorphisms and CanLab (and hash).
#Load graph: "n=5 g 1:2,5 2:1,3 3:2,4 4:3,5 5:1,4."
#Set Fixed Points: "-f -F=x1 -F=x2 ..."
#Add Fixed Points: "-F x1"
#Automorphisms: "x"
#CanLab: "b"
#Hash: "z"

################# Sample Session:
# $dreadnaut 
# Dreadnaut version 2.7 (64 bits).
# > l=0$=1dac
# > n=5 g 1:2,5 2:1,3 3:2,4 4:3,5 5:1,4.
# > x
# (2 5)(3 4)
# level 2:  3 orbits; 2 fixed; index 2
# (1 2)(3 5)
# level 1:  1 orbit; 1 fixed; index 5
# 1 orbit; grpsize=10; 2 gens; 6 nodes; maxlev=3
# canupdates=1; cpu time = 0.00 seconds
# > b 
#  1 2 5 3 4
#   1 :  2 3;
#   2 :  1 4;
#   3 :  1 5;
#   4 :  2 5;
#   5 :  3 4;
# > z
# [Nc5d7751 53d3d98b c052e6a]
# > -F=3
# > x
# [fixing partition]
# (1 5)(2 4)
# level 1:  3 orbits; 2 fixed; index 2
# 3 orbits; grpsize=2; 1 gen; 3 nodes; maxlev=2
# canupdates=1; cpu time = 0.00 seconds
# > -f
# > -F=1
# > x
# [fixing partition]
# (2 5)(3 4)
# level 1:  3 orbits; 2 fixed; index 2
# 3 orbits; grpsize=2; 1 gen; 3 nodes; maxlev=2
# canupdates=1; cpu time = 0.00 seconds
# > -F=2
# > x
# [fixing partition]
# 5 orbits; grpsize=1; 0 gens; 1 node; maxlev=1
# canupdates=1; cpu time = 0.00 seconds
# > q

